package de.ugoe.cs.autoquest.eventcore;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * Base class for all events. An event is described by its {@link #type} and its {@link #target}.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 * 
 * @param <T>
 *            Can be used to declare that events belong to a specific platform without subclassing.
 */
public class Event implements Serializable {

    /**
     * <p>
     * Id for object serialization.
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * Global start event that can be used to indicate the start of a sequence.
     * </p>
     */
    public static final Event STARTEVENT = new Event(new StringEventType("START"));

    /**
     * <p>
     * Global end event that can be used to indicate the end of a sequence.
     */
    public static final Event ENDEVENT = new Event(new StringEventType("END"));

    /**
     * <p>
     * Type of the event.
     * </p>
     */
    protected IEventType type;

    /**
     * </p> Target of the event.
     */
    protected IEventTarget target = null;

    /**
     * <p>
     * List of {@link IReplayable}s of type T that describes the replay of an event. The
     * {@link IReplayable}s can be interpreted as <it>sub-events</it> on the platform level that
     * make up the abstract event.
     * </p>
     */
    protected List<IReplayable> replay = new LinkedList<IReplayable>();

    /**
     * <p>
     * Constructor. Creates a new Event with a given type. The type must not be null.
     * </p>
     * 
     * @param type
     *            type of the event
     */
    public Event(IEventType type) {
        if (type == null) {
            throw new IllegalArgumentException("Event type must not be null");
        }
        this.type = type;
    }

    /**
     * <p>
     * Constructor. Creates a new Event with a given type and target. The type must not be null.
     * </p>
     * 
     * @param type
     *            type of the event
     * @param target
     *            target of the event
     */
    public Event(IEventType type, IEventTarget target) {
        this(type);
        this.target = target;
    }

    /**
     * <p>
     * Two events are equal, if their {@link #type} and {@link #target} are equal.
     * </p>
     * <p>
     * See {@link Object#equals(Object)} for further information.
     * </p>
     * 
     * @param other
     *            Event that is compared to this
     * @return true, if events are equal, false otherwise
     */
    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other instanceof Event) {
            Event otherEvent = (Event) other;
            if (target != null) {
                return type.equals(otherEvent.type) && target.equals(otherEvent.target);
            }
            else {
                return type.equals(otherEvent.type) && otherEvent.target == null;
            }
        }
        return false;
    }

    /**
     * <p>
     * Returns {@link #getId()} as String representation of the event.
     * </p>
     * 
     * @return String representation of the event
     */
    @Override
    public String toString() {
        return getId();
    }

    /**
     * <p>
     * Returns the {@link #target} of the event.
     * </p>
     * 
     * @return {@link #target} of the event
     */
    public IEventTarget getTarget() {
        return target;
    }

    /**
     * <p>
     * Returns the {@link #type} of the event.
     * </p>
     * 
     * @return {@link #type} of the event
     */
    public IEventType getType() {
        return type;
    }
    
    /**
     * <p>
     * Returns the combination of {@link #type} and {@link #target} as id.
     * </p>
     *
     * @return string of the form (type,target)
     */
    public String getId() {
        String id = type.toString();
        if( target!=null ) {
            id += "." + target.getStringIdentifier();
        }
        return id;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        int multiplier = 17;
        int hash = 42;
        if (type != null) {
            hash = multiplier * hash + type.hashCode();
        }
        if (target != null) {
            hash = multiplier * hash + target.hashCode();
        }
        return hash;
    }

    /**
     * <p>
     * Sets the target of the event. Once set, the target cannot be changed.
     * </p>
     * 
     * @param target
     *            target of the event
     * @return true, if target was changed, false otherwise
     */
    public boolean setTarget(IEventTarget target) {
        if (this.target != null) {
            return false;
        }
        this.target = target;
        return true;
    }

    /**
     * <p>
     * Adds a new {@link IReplayable} of type T to the replay sequence.
     * </p>
     * 
     * @param replayable
     *            element that is added to the sequence
     * @throws IllegalArgumentException
     *             thrown is replayable is null
     */
    public void addReplayable(IReplayable replayable) {
        if (replayable == null) {
            throw new IllegalArgumentException("replayble must not be null");
        }
        replay.add(replayable);
    }

    /**
     * <p>
     * Adds a {@link List}ist of {@link IReplayable} to the replay sequence.
     * </p>
     * 
     * @param generatedReplaySeq
     *            {@link List} that is added to the sequence
     * @throws IllegalArgumentException
     *             thrown if generatedReplaySeq is null
     */
    public void addReplayableSequence(List<? extends IReplayable> generatedReplaySeq) {
        if (generatedReplaySeq == null) {
            throw new IllegalArgumentException("generatedReplaySeq must not be null");
        }
        replay.addAll(generatedReplaySeq);
    }

    /**
     * <p>
     * Returns a the list of replay events.
     * </p>
     * <p>
     * The return value is a copy of the list used internally!
     * </p>
     * 
     * @return list of replay events.
     */
    public List<IReplayable> getReplayables() {
        return new LinkedList<IReplayable>(replay);
    }
}
