package de.ugoe.cs.autoquest.tasktrees.nodeequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * This class is capable of comparing any task tree node which is not an iteration with an
 * iteration. This is needed, because iterations may iterate exactly that node. In this
 * case, the iteration would be equal to that node if it was executed exactly once. The rule
 * returns lexically equal, it the child of the iteration is lexically equal to the node
 * or if the child of the iteration is a selection and this selections contains a lexically equal
 * node. The same applies for syntactical and semantical equality.
 * </p>

 * @author Patrick Harms
 */
public class NodeAndIterationComparisonRule implements NodeComparisonRule {
    
    /** the rule manager for internally comparing task tree nodes */
    private NodeEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the node equality rule manager to be able
     * to perform comparisons of the children of provided task tree nodes
     * </p>
     * 
     * @param ruleManager the rule manager for comparing task tree nodes
     */
    NodeAndIterationComparisonRule(NodeEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.tasktree.nodeequality.NodeEqualityRule#apply(TaskTreeNode, TaskTreeNode)
     */
    @Override
    public NodeEquality compare(ITaskTreeNode node1, ITaskTreeNode node2) {
        IIteration iteration = null;
        ITaskTreeNode node = null;
        
        if (node1 instanceof IIteration) {
            if (node2 instanceof IIteration) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            iteration = (IIteration) node1;
            node = node2;
        }
        else if (node2 instanceof IIteration) {
            if (node1 instanceof IIteration) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            iteration = (IIteration) node2;
            node = node1;
        }
        else {
            return null;
        }

        // now, that we found the iteration and the node, lets compare the child of the iteration
        // with the node.
        if (iteration.getChildren().size() < 1) {
            return null;
        }

        NodeEquality nodeEquality = mRuleManager.applyRules(iteration.getChildren().get(0), node);

        // although the subtask may be identical to the node, we can not return identical, as
        // the iteration is not identical to the node, but at most lexically equal
        if (nodeEquality == NodeEquality.IDENTICAL) {
            return NodeEquality.LEXICALLY_EQUAL;
        }
        else {
            return nodeEquality;
        }

    }
}
