
package de.ugoe.cs.autoquest.plugin.mfc.eventcore;

import java.util.Map;

import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.gui.IInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.KeyPressed;
import de.ugoe.cs.autoquest.eventcore.gui.KeyReleased;
import de.ugoe.cs.autoquest.eventcore.gui.KeyboardFocusChange;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.MouseClick;
import de.ugoe.cs.autoquest.eventcore.gui.ValueSelection;
import de.ugoe.cs.autoquest.keyboardmaps.VirtualKey;

/**
 * <p>
 * Creates the GUI event types (i.e., {@link IInteraction}s) for MFC events.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public class MFCEventTypeFactory {

    /**
     * <p>
     * Instance of the singleton
     * </p>
     */
    private static MFCEventTypeFactory instance = new MFCEventTypeFactory();

    /**
     * <p>
     * Constructor. Creates a new MFCEventTypeFactory. Private to preserve singleton property.
     * </p>
     * 
     */
    private MFCEventTypeFactory() {}

    /**
     * <p>
     * Returns the instance of the MFCEventTypeFactory.
     * </p>
     * 
     * @return the instance
     */
    public static MFCEventTypeFactory getInstance() {
        return instance;
    }

    /**
     * <p>
     * Returns the event type based on the name and parameters of a MFC event.
     * </p>
     * 
     * @param eventName
     *            name of the MFC event
     * @param messageParameters
     *            parameters of the MFC event
     * @return the event type
     */
    public IEventType getEventType(String eventName, Map<String, String> messageParameters) {
        if ("LeftClickButton".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("LeftClickListBox".equals(eventName)) {
            return new ValueSelection<Integer>(getSelectedValue(messageParameters));
        }
        else if ("TabChange".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("LeftClickCommand".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("LeftClickSysCommand".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("NCLeftClickSysCommand".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("LeftClickMenuItemCmd".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("HScroll_TrackBar".equals(eventName)) {
            return new ValueSelection<Integer>(getSelectedValue(messageParameters));
        }
        else if ("VScroll_TrackBar".equals(eventName)) {
            return new ValueSelection<Integer>(getSelectedValue(messageParameters));
        }
        else if ("HScroll_ScrollBar".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("VScroll_ScrollBar".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("VScrollNC".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("LeftClickSetFocus".equals(eventName)) {
            return new KeyboardFocusChange();
        }
        else if ("LeftClickChangeFocus".equals(eventName)) {
            return new KeyboardFocusChange();
        }
        else if ("KeyDown".equals(eventName)) {
            return new KeyPressed(getKey(messageParameters));
        }
        else if ("KeyUp".equals(eventName)) {
            return new KeyReleased(getKey(messageParameters));
        }
        else if ("SysKeyDown".equals(eventName)) {
            return new KeyPressed(getKey(messageParameters));
        }
        else if ("SysKeyUp".equals(eventName)) {
            return new KeyReleased(getKey(messageParameters));
        }
        else if ("LeftClickCoordinates".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("NCLeftClickCoordinates".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("NCLeftClickCoordinates2".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("LeftClickCoordinatesTargetChanged".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else if ("LeftClickCoordinatesTargetChanged2".equals(eventName)) {
            return new MouseClick(MouseButtonInteraction.Button.LEFT);
        }
        else {
            throw new IllegalArgumentException("unknown event name: " + eventName);
        }
    }

    /**
     * <p>
     * If the message parameters contain information about a key that has been pressed, the
     * associated {@link VirtualKey} is returned.
     * </p>
     * 
     * @param messageParameters
     *            the message parameters
     * @return key extracted from the message parameters
     * @throws IllegalArgumentException
     *             thrown if the messageParameters do not contain information about a key
     */
    private VirtualKey getKey(Map<String, String> messageParameters)
        throws IllegalArgumentException
    {
        String value = null;

        if (messageParameters != null) {
            value = messageParameters.get("key");
        }

        if (value == null) {
            throw new IllegalArgumentException
                ("no parameter \"key\" provided for key event. Please correct the event " +
                 "generation rules");
        }
        
        return WindowsVirtualKey.parseVirtualKey(value).getKey();
    }

    /**
     * <p>
     * If the message parameters contain information about a scroll position, the respective
     * position is returned.
     * </p>
     * 
     * @param messageParameters
     *            the message parameters
     * @return the scroll position
     * @throws IllegalArgumentException
     *             thrown if the messageParameters do not contain information about a scroll
     *             position
     */
    private int getSelectedValue(Map<String, String> messageParameters)
        throws IllegalArgumentException
    {
        String value = null;

        if (messageParameters != null) {
            value = messageParameters.get("scrollPos");
        }

        if (value == null) {
            throw new IllegalArgumentException
                ("no parameter \"scrollPos\" provided for scroll event. Please correct the event " +
                 "generation rules");
        }
        
        return Integer.parseInt(value);
    }

}
