//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.assertions;

import de.ugoe.cs.autoquest.IReplayDecorator;
import de.ugoe.cs.autoquest.eventcore.IReplayable;
import de.ugoe.cs.util.StringTools;

/**
 * <p>
 * This class defines the replay for a textEquals assertion.
 * </p>
 * 
 * @author Jeffrey Hall, Steffen Herbold
 * @version 2.0
 */
public class TextEqualsReplay implements IReplayable {

    /**
     * <p>
     * Reference value which is compared to the targets text.
     * </p>
     */
    protected final String expectedValue;

    /**
     * <p>
     * Target to which the text is compared.
     * </p>
     */
    protected final String target;

    /**
     * <p>
     * Id for object serialization.
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * Constructor. Creates a new TextEqualsReplay.
     * 
     * @param expectedValue
     *            expected string value
     * @param target
     *            string description of the target whose string value is compared to the expected
     *            value
     * @throws IllegalArgumentException
     *             thrown if target is null
     */
    public TextEqualsReplay(String expectedValue, String target) {
        if (target == null) {
            throw new IllegalArgumentException("target must not be null");
        }
        this.expectedValue = expectedValue;
        this.target = target;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.eventcore.IReplayable#getReplay()
     */
    @Override
    public String getReplay() {

        String expectedValueTmp = StringTools.xmlEntityReplacement(expectedValue);

        StringBuilder currentMsgStr = new StringBuilder(400);
        currentMsgStr.append(" <textEquals expectedValue=\"" + expectedValueTmp + "\">");
        currentMsgStr.append(StringTools.ENDLINE);
        currentMsgStr.append("<target>");
        currentMsgStr.append(StringTools.ENDLINE);
        currentMsgStr.append(target);
        currentMsgStr.append(StringTools.ENDLINE);
        currentMsgStr.append("</target>");
        currentMsgStr.append(StringTools.ENDLINE);
        currentMsgStr.append("</textEquals>");
        currentMsgStr.append(StringTools.ENDLINE);
        return currentMsgStr.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.eventcore.IReplayable#getDecorator()
     */
    @Override
    public IReplayDecorator getDecorator() {
        return null;
    }

}
