//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin;

import java.io.File;
import java.util.Arrays;
import java.util.Collection;

import junitx.framework.ArrayAssert;

import org.junit.*;

import de.ugoe.cs.autoquest.plugin.PluginLoader;
import de.ugoe.cs.autoquest.plugin.PluginLoaderException;
import de.ugoe.cs.autoquest.plugin.QuestPlugin;
import static org.junit.Assert.*;

/**
 * The class <code>PluginLoaderTest</code> contains tests for the class
 * <code>{@link PluginLoader}</code>.
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class PluginLoaderTest {
	
	@Test
	public void testPluginLoader_1() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		assertNotNull(loader);
	}
	
	@Test(expected = java.lang.IllegalArgumentException.class)
	public void testPluginLoader_2() throws Exception {
		new PluginLoader(null);
	}
	
	@Test(expected = java.lang.IllegalArgumentException.class)
	public void testPluginLoader_3() throws Exception {
		new PluginLoader(new File("testdata/de.ugoe.cs.autoquest.plugin.PluginLoaderTest/jfcmonitor.jar"));
	}
		
	@Test
	public void testCheckNameConformity_1() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "quest-plugin-jfc-1.0.jar";
		boolean expected = true;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_2() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "quest-plugin-jf-c-1.0.jar";
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_3() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "quest-plugin-jfc.jar";
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_4() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "quest-plugi-jfc-1.0.jar";
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_5() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "quest-pluginjfc-1.0.jar";
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_6() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "quest-plugin-jfc-1-0.jar";
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_7() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "quest-plugin-jfc-1.0.nojar";
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_8() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = null;
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testCheckNameConformity_9() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		String filename = "";
		boolean expected = false;
		
		boolean result = loader.checkNameConformity(filename);
		
		assertEquals(expected, result);
	}
	
	@Test
	public void testGetClassPathFromJar_1() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		File jarFile = new File("testdata/de.ugoe.cs.autoquest.plugin.PluginLoaderTest/jfcmonitor.jar");
		
		String[] expected = new String[]{ "file:" + jarFile.getParentFile().getAbsolutePath()+"/javahelperlib.jar" };
				
		String[] result = loader.getClassPathFromJar(jarFile);
		
		ArrayAssert.assertEquivalenceArrays(expected, result);
	}
	
	@Test
	public void testGetClassPathFromJar_2() throws Exception {
		PluginLoader loader = new PluginLoader(new File("."));
		File jarFile = new File("testdata/de.ugoe.cs.autoquest.plugin.PluginLoaderTest/jmi.jar");
		
		String[] expected = new String[]{ };
				
		String[] result = loader.getClassPathFromJar(jarFile);
		
		ArrayAssert.assertEquivalenceArrays(expected, result);
	}
	
	@Test 
	public void testLoad_1() throws Exception {
		PluginLoader loader = new PluginLoader(new File("testdata/de.ugoe.cs.autoquest.plugin.PluginLoaderTest"));
		
		loader.load();
		
		Collection<QuestPlugin> plugins = loader.getPlugins();
		
		assertEquals(1, plugins.size());
		QuestPlugin plugin = plugins.iterator().next();
		assertNotNull(plugin);
		assertEquals("Mock Plugin", plugin.getTitle());
		assertEquals(Arrays.asList(new String[]{"de.ugoe.cs.autoquest.plugin.mock.commands"}), plugin.getCommandPackages());
	}
	
	@Test 
	public void testLoad_2() throws Exception {
		PluginLoader loader = new PluginLoader(new File("testdata/de.ugoe.cs.autoquest.plugin.PluginLoaderTestInvalid_1"));
		
		try {
			loader.load();
		} catch(PluginLoaderException e) {
			e.getMessage().endsWith("not instance of QuestPlugin");
		}
	}
	
	@Test 
	public void testLoad_3() throws Exception {
		PluginLoader loader = new PluginLoader(new File("testdata/de.ugoe.cs.autoquest.plugin.PluginLoaderTestInvalid_2"));
		
		try {
			loader.load();
		} catch(PluginLoaderException e) {
			e.getMessage().startsWith("No class");
		}
	}
	
	@Test 
	public void testLoad_4() throws Exception {
		PluginLoader loader = new PluginLoader(new File("testdata/de.ugoe.cs.autoquest.plugin.PluginLoaderTestInvalid_3"));
		
		try {
			loader.load();
		} catch(PluginLoaderException e) {
			e.getMessage().endsWith("Could not access");
		}
	}

}
