//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.eventcore.gui;

import java.util.LinkedList;
import java.util.List;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.IEventTarget;

/**
 * <p>
 * This class condenses mouse clicks, i.e. it reduces a sequence of mouse button down, mouse button
 * up and mouse click with the same button on the same event target to a single mouse click with
 * that button on that target. The mouse button down and mouse button up events are discarded. For
 * this, it iterates the provided sequence and identifies any match of the named event sequence
 * pattern. This match is condensed to the mouse click event.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public class MouseClickCondenser {

    /**
     * <p>
     * This method performs the work described in the description of the class. A new list is
     * instantiated and returned. This list is filled with the events provided by the sequence being
     * the parameter of the method except for mouse button down and mouse button up events which are
     * followed by a mouse click event with the same button on the same target.
     * </p>
     * 
     * @param sequence
     *            the event sequence to condense the mouse clicks in
     * 
     * @return the resulting sequence, in which mouse clicks are condensed to single mouse click
     *         events
     */
    public List<Event> condenseMouseClicks(List<Event> sequence) {
        List<Event> resultingSequence = new LinkedList<Event>();

        int index = 0;
        while (index < sequence.size()) // -2 because we don't need to go to the end
        {
            if ((index + 2) < sequence.size()) {
                Event mouseButtonDown = sequence.get(index);
                Event mouseButtonUp = sequence.get(index + 1);
                Event mouseClick = sequence.get(index + 2);
                if (mouseClickSequenceFound(mouseButtonDown, mouseButtonUp, mouseClick)) {
                    // skip the mouse button down and mouse button up event
                    index += 2;
                }
            }

            resultingSequence.add(sequence.get(index));
            index++;
        }

        return resultingSequence;
    }

    /**
     * 
     */
    private boolean mouseClickSequenceFound(Event mouseButtonDown,
                                            Event mouseButtonUp,
                                            Event mouseClick)
    {
        // check the first in a row of three for validity
        if (!(mouseButtonDown.getType() instanceof MouseButtonDown)) {
            return false;
        }

        // check the second node for validity
        if (!(mouseButtonUp.getType() instanceof MouseButtonUp)) {
            return false;
        }

        IEventTarget eventTarget = mouseButtonDown.getTarget();

        if (!eventTarget.equals(mouseButtonUp.getTarget())) {
            return false;
        }

        MouseButtonInteraction.Button button =
            ((MouseButtonDown) mouseButtonDown.getType()).getButton();

        if (!button.equals(((MouseButtonUp) mouseButtonUp.getType()).getButton())) {
            return false;
        }

        // check the third node for validity
        if (!(mouseClick.getType() instanceof MouseClick)) {
            return false;
        }

        if (!eventTarget.equals(mouseClick.getTarget())) {
            return false;
        }

        if (!button.equals(((MouseClick) mouseClick.getType()).getButton())) {
            return false;
        }

        return true;
    }

}
