package de.ugoe.cs.autoquest.tasktrees.nodeequality;

import de.ugoe.cs.autoquest.eventcore.gui.IInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.TextInput;
import de.ugoe.cs.autoquest.eventcore.gui.ValueSelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * This rule compares GUI event tasks (i.e. it is more concrete, than the
 * {@link EventTaskComparisonRule}). Two GUI event tasks are only equal if their event type and
 * target are equal. The returned equality is even more fine-grained for events whose type is
 * {@link TextInput} and {@link ValueSelection}. For text inputs, lexical equality is returned if
 * the same text is entered using the same key interactions. Syntactical equality is returned if
 * the same text is entered using different key interactions. Semantical equality is returned if
 * different text is entered, but into the same event target. Value selections are syntactically
 * equal, if the same value is selected. Otherwise they are semantically equal.
 * </p>
 * 
 * @author Patrick Harms
 */
public class GUIEventTaskComparisonRule implements NodeComparisonRule {
    
    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.tasktree.nodeequality.NodeEqualityRule#apply(TaskTreeNode, TaskTreeNode)
     */
    @Override
    public NodeEquality compare(ITaskTreeNode node1, ITaskTreeNode node2) {
        if ((!(node1 instanceof IEventTask)) || (!(node2 instanceof IEventTask))) {
            return null;
        }
        
        IEventTask task1 = (IEventTask) node1;
        IEventTask task2 = (IEventTask) node2;
        
        if ((!(task1.getEventType() instanceof IInteraction)) ||
            (!(task2.getEventType() instanceof IInteraction)))
        {
            return null;
        }

        if (node1 == node2) {
            return NodeEquality.IDENTICAL;
        }

        if (!task1.getEventTarget().equals(task2.getEventTarget())) {
            return NodeEquality.UNEQUAL;
        }
        
        IInteraction interaction1 = (IInteraction) task1.getEventType();
        IInteraction interaction2 = (IInteraction) task2.getEventType();
        
        return compareInteractions(interaction1, interaction2);
    }

    /**
     * <p>
     * compares two interactions. The method delegates two
     * {@link #compareTextInputs(TextInput, TextInput)} and
     * {@link #compareValueSelections(ValueSelection, ValueSelection)} for text inputs and value
     * selections. Otherwise it uses the equal method of the interactions for comparison. In this
     * case, if the interactions equal method returns true, this method returns lexical equality.
     * </p>
     *
     * @param interaction1 the first interaction to compare
     * @param interaction2 the second interaction to compare
     * 
     * @return as described
     */
    private NodeEquality compareInteractions(IInteraction interaction1, IInteraction interaction2) {
        if (interaction1 == interaction2) {
            return NodeEquality.LEXICALLY_EQUAL;
        }
        else if ((interaction1 instanceof TextInput) && (interaction2 instanceof TextInput)) {
            return compareTextInputs((TextInput) interaction1, (TextInput) interaction2);
        }
        else if ((interaction1 instanceof ValueSelection) &&
                 (interaction2 instanceof ValueSelection))
        {
            return compareValueSelections
                ((ValueSelection<?>) interaction1, (ValueSelection<?>) interaction2);
        }
        else if (interaction1.equals(interaction2)) {
            return NodeEquality.LEXICALLY_EQUAL;
        }
        else {
            return NodeEquality.UNEQUAL;
        }
    }

    /**
     * <p>
     * compares two text inputs. If both text inputs have the same entered text and text input
     * events, they are lexically equal. If they only have the same entered text, they are
     * syntactically equal. If they are only both text inputs, they are semantically equal.
     * (the equality of the event targets is checked beforehand).
     * </p>
     *
     * @param interaction1 the first text input to compare
     * @param interaction2 the second text input to compare
     * 
     * @return as described
     */
    private NodeEquality compareTextInputs(TextInput interaction1, TextInput interaction2) {
        if (interaction1.getEnteredText().equals(interaction2.getEnteredText())) {
            if (interaction1.getTextInputEvents().equals(interaction2.getTextInputEvents())) {
                return NodeEquality.LEXICALLY_EQUAL;
            }
            else {
                return NodeEquality.SYNTACTICALLY_EQUAL;
            }
        }
        else {
            return NodeEquality.SEMANTICALLY_EQUAL;
        }
    }

    /**
     * <p>
     * compares two value selections. If both value selections have the same selected value, they
     * are syntactically equal, otherwise they are semantically equal.
     * (the equality of the event targets is checked beforehand).
     * </p>
     *
     * @param interaction1 the first value selection to compare
     * @param interaction2 the second value selection to compare
     * 
     * @return as described
     */
    private NodeEquality compareValueSelections(ValueSelection<?> interaction1,
                                                ValueSelection<?> interaction2)
    {
        Object value1 = interaction1.getSelectedValue();
        Object value2 = interaction2.getSelectedValue();
        
        if ((value1 == value2) || ((value1 != null) && (value1.equals(value2)))) {
            return NodeEquality.SYNTACTICALLY_EQUAL;
        }
        else {
            return NodeEquality.SEMANTICALLY_EQUAL;
        }
    }

}
