//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.guimodel;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.ugoe.cs.autoquest.eventcore.guimodel.GUIElementFactory;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModelException;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementFactory;

/**
 * <p>
 * This class provides an the interfaces for component trees.
 * </p>
 * <p>
 * The component tree represents the hierarchical structure of the components "as it is" currently during
 * a session. It may change during the session due to creation and destruction of components.
 * </p>
 * 
 * @author Steffen Herbold
 * @author Fabian Glaser
 * @version 1.0
 */
public class JFCComponentTree {

    /**
     * <p>
     * Map of all GUI element specifications that are part of the tree for efficient searching. The
     * keys of the map are the hash values of the GUI elements.
     * </p>
     */
    private Map<Integer, JFCGUIElementSpec> guiElementSpecs;

    /**
     * <p>
     * Map of all children of GUI elements that are part of the tree. The keys of the map are the
     * hash values of the parent GUI elements.
     * </p>
     */
    private Map<Integer, List<JFCGUIElementSpec>> childRelations;

    /**
     * <p>
     * Map of all parents of GUI elements that are part of the tree. The keys of the map are the
     * hash values of the child GUI elements.
     * </p>
     */
    private Map<Integer, JFCGUIElementSpec> parentRelations;

    /**
     * <p>
     * the internally created GUI model
     * </p>
     */
    private GUIModel guiModel = new GUIModel();

    /**
     * <p>
     * the GUI element factory used in the model
     * </p>
     */
    private IGUIElementFactory guiElementFactory = GUIElementFactory.getInstance();

    /**
     * <p>
     * Map of all GUI elements that are part of the tree for efficient searching. The keys of the
     * map are the hash values of the GUI elements.
     * </p>
     */
    private Map<Integer, JFCGUIElement> guiElements;

    /**
     * <p>
     * Creates a new JFCComponentTree.
     * </p>
     */
    public JFCComponentTree() {
        guiElementSpecs = new HashMap<Integer, JFCGUIElementSpec>();
        childRelations = new HashMap<Integer, List<JFCGUIElementSpec>>();
        parentRelations = new HashMap<Integer, JFCGUIElementSpec>();
        guiElements = new HashMap<Integer, JFCGUIElement>();
    }

    /**
     * <p>
     * Adds a new component to the tree.
     * </p>
     * 
     * @param componentHash
     *            hash of the window to be created
     * @param parentHash
     *            hash of the parent window
     * @param componentSpec
     * 			  the component specification
     */
    public void add(Integer componentHash,
    				Integer parentHash,
                    JFCGUIElementSpec componentSpec)
    {
        JFCGUIElement guiElement = guiElements.get(componentHash);
        
        if (guiElement == null) {
        	JFCGUIElementSpec parent = guiElementSpecs.get(parentHash);
            if (parent != null) {
                List<JFCGUIElementSpec> otherChildren = childRelations.get(parentHash);

                if (otherChildren == null) {
                    otherChildren = new ArrayList<JFCGUIElementSpec>();
                    childRelations.put(parentHash, otherChildren);
                }

                otherChildren.add(componentSpec);

                parentRelations.put(componentHash, parent);
            }
            guiElementSpecs.put(componentHash, componentSpec);
            
            List<JFCGUIElementSpec> guiElementPath = new ArrayList<JFCGUIElementSpec>();

            while (componentSpec != null) {
                guiElementPath.add(0, componentSpec);
                componentSpec = parentRelations.get(componentSpec.getElementHash());
            }

            try {
                guiElement =
                    (JFCGUIElement) guiModel.integratePath(guiElementPath, guiElementFactory);
            }
            catch (GUIModelException e) {
                throw new RuntimeException("could not instantiate GUI element with id " + componentHash, e);
            }
            guiElements.put(componentHash, guiElement);
        }
    }

    /**
     * <p>
     * Searches the tree for a component with the specified hash and returns its
     * {@link JFCGUIElementSpec} .
     * </p>
     * 
     * @param hash
     *            hash that is looked for
     * @return {@link JFCGUIElementSpec} of the window with the given hash if found, null otherwise
     */
    public JFCGUIElement find(int hash) {
        JFCGUIElement guiElement = guiElements.get(hash);
        if (guiElement == null) {
            List<JFCGUIElementSpec> guiElementPath = new ArrayList<JFCGUIElementSpec>();

            JFCGUIElementSpec child = guiElementSpecs.get(hash);

            if (child == null) {
                throw new RuntimeException("no GUI element found with hash " + hash);
            }

            while (child != null) {
                guiElementPath.add(0, child);
                child = parentRelations.get(child.getElementHash());
            }

            try {
                guiElement =
                    (JFCGUIElement) guiModel.integratePath(guiElementPath, guiElementFactory);
            }
            catch (GUIModelException e) {
                throw new RuntimeException("could not instantiate GUI element with id " + hash, e);
            }
            guiElements.put(hash, guiElement);
        }
        return guiElement;
    }

    /**
     * <p>
     * Sets the name of a GUI element given its hash.
     * </p>
     * 
     * @param hash
     *            hash of the GUI element
     * @param windowName
     *            new name of the GUI element
     */
    public void setName(int hash, String windowName) {
        JFCGUIElementSpec child = guiElementSpecs.get(hash);
        if (child != null) {
            child.setName(windowName);

            JFCGUIElement guiElement = guiElements.remove(hash);
            if (guiElement == null) {
                // we need to update the GUI model as well
                find(hash);
            }
        }
    }

    /**
     * <p>
     * Removes a window (defined by its hash) from the tree. All children of the window will be
     * removed recursively.
     * </p>
     * 
     * @param hash
     *            hash of the window to be removed
     * @return number of windows that were removed
     */
    public int remove(long hash) {
        JFCGUIElementSpec node = guiElementSpecs.remove(hash);
        int removedCounter = 1;

        if (node != null) {
            List<JFCGUIElementSpec> nodesToBeRemoved = childRelations.remove(hash);

            // remove all children and sub-children, if any
            if (nodesToBeRemoved != null) {
                for (int i = 0; i < nodesToBeRemoved.size(); i++) {
                    JFCGUIElementSpec nodeToBeRemoved = nodesToBeRemoved.get(i);
                    List<JFCGUIElementSpec> children =
                        childRelations.remove(nodeToBeRemoved.getElementHash());

                    if (children != null) {
                        nodesToBeRemoved.addAll(children);
                    }

                    guiElementSpecs.remove(nodeToBeRemoved.getElementHash());
                    parentRelations.remove(nodeToBeRemoved.getElementHash());
                    removedCounter++;
                }
            }

            // the node may be a child node of a parent. So search for it and remove it
            JFCGUIElementSpec parent = parentRelations.remove(hash);
            if (parent != null) {
                List<JFCGUIElementSpec> children = childRelations.get(parent.getElementHash());

                if (children != null) {
                    for (int i = 0; i < children.size(); i++) {
                        if (children.get(i).getElementHash() == hash) {
                            children.remove(i);
                            break;
                        }
                    }

                    if (children.size() <= 0) {
                        childRelations.remove(parent.getElementHash());
                    }
                }
            }
        }
        return removedCounter;
    }

    /**
     * @return the guiModel
     */
    public GUIModel getGUIModel() {
        return guiModel;
    }

    /**
     * <p>
     * Returns the number of nodes contained in the JFCComponentTree.
     * </p>
     * 
     * @return number of nodes
     */
    public int size() {
        return guiElementSpecs.size();
    }

}
